import videojs from "video.js";
import shaka from "shaka-player";
import { setupShakaDRM } from "./shaka-drm-config";
import setupQualityTracks from "./setup-quality-tracks";
import setupTextTracks from "./setup-text-tracks";
import setupAudioTracks from "./setup-audio-tracks";

const Html5 = videojs.getTech("Html5");

// Expose Shaka library globally (only once)
if (typeof window !== "undefined" && !window.__shaka) {
  window.__shaka = shaka;
  console.log("[Shaka] Exposed Shaka library to window.__shaka");
  // 3.1 Log Shaka version (shared across all instances)
  console.log("[Shaka] Version:", shaka.Player.version);
}

class Shaka extends Html5 {
  constructor(options, ready) {
    super(options, ready);

    this.vjsPlayer = videojs(options.playerId);
    this.shaka_ = null;

    this.player_.ready(() => {
      this.player_.addClass("vjs-shaka");
    });
  }

  static isSupported() {
    return !!window.MediaSource && shaka.Player.isBrowserSupported();
  }

  static canPlaySource(sourceObj, type) {
    const dashTypeRE = /^(application\/dash\+xml|application\/x-mpegURL)/i;

    if (dashTypeRE.test(sourceObj.type)) {
      return "probably";
    }

    return "";
  }

  createEl() {
    this.el_ = Html5.prototype.createEl.apply(this, arguments);

    // Install built-in polyfills to patch browser incompatibilities.
    shaka.polyfill.installAll();

    // Set debug log level
    if (shaka.log) {
      if (this.options_.debug) {
        shaka.log.setLevel(shaka.log.Level.DEBUG);
      } else {
        shaka.log.setLevel(shaka.log.Level.ERROR);
      }
    }

    this.el_.tech = this;

    return this.el_;
  }

  // Helper method to get container tag for logging
  getContainerTag() {
    return this.containerId_ ? `[${this.containerId_}]` : "";
  }

  async initializeShaka() {
    const containerTag = this.getContainerTag();
    
    if (!this.shaka_ && this.el_) {
      console.log(`${containerTag} [Shaka] Initializing Shaka Player...`);
      this.shaka_ = new shaka.Player();
      await this.shaka_.attach(this.el_);
      
      // Expose Shaka player instance if containerId is available
      if (this.containerId_) {
        this.exposeShakaInstances();
      }
      // this.shaka_.configure({
      //   streaming: {
      //     useNativeHlsForFairPlay: false,
      //     preferNativeHls: false,
      //   },
      // });

      // 3.2 Log ABR config + restrictions after configuration
      console.log(`${containerTag} [Shaka] Effective config abr:`, this.shaka_.getConfiguration().abr);
      console.log(`${containerTag} [Shaka] Effective config restrictions:`, this.shaka_.getConfiguration().restrictions);
      
      console.log(`${containerTag} [Shaka] Player initialized`);
    } else if (!this.el_) {
      console.log(`${containerTag} [Shaka] Element not created, cannot initialize Shaka Player`);
    }
  }

  async setSrc(src) {
    const me = this;

    // Ensure element and Shaka Player are initialized
    if (!this.el_) {
      console.log("Element has not been created, waiting...");
      // Add retry up to 3 times when waiting for this.el_
      if (typeof this._elRetryCount === "undefined") {
        this._elRetryCount = 0;
      }
      this._elRetryCount++;
      if (this._elRetryCount > 3) {
        console.error(
          "Cannot create element for Shaka Player after 3 retries."
        );
        return;
      }
      setTimeout(() => {
        me.setSrc(src);
      }, 100);
      return;
    }

    if (!this.shaka_) {
      console.log("Shaka Player has not been initialized, initializing...");
      await this.initializeShaka();
    }

    if (!this.shaka_) {
      console.log("Cannot initialize Shaka Player");
      return;
    }

    const shakaOptions = this.options_.configuration || {};

    if (!shakaOptions.abr) {
      shakaOptions.abr = {
        enabled: true,
      };
    }

    if (this.options_.drmConfig) {
      await setupShakaDRM(this.shaka_, this.options_.drmConfig);
    }

    const containerTag = this.getContainerTag();
    
    // 3.1 Log load request
    console.log(`${containerTag} [Shaka] Load request:`, { 
      uri: src, 
      isHls: src.includes('.m3u8') 
    });

    // 3.2 Log ABR config + restrictions before load
    console.log(`${containerTag} [Shaka] Effective config abr:`, this.shaka_.getConfiguration().abr);
    console.log(`${containerTag} [Shaka] Effective config restrictions:`, this.shaka_.getConfiguration().restrictions);

    this.shaka_.addEventListener("buffering", function (event) {
      if (event.buffering) {
        me.vjsPlayer.trigger("waiting");
      } else {
        me.vjsPlayer.trigger("playing");
      }
    });

    // 3.5 Error logging
    this.shaka_.addEventListener("error", function (event) {
      const containerTag = me.getContainerTag();
      console.error(`${containerTag} [Shaka] Player error event:`, event.detail);
      me.retriggerError(event.detail);
    });

    // 3.5 Adaptation event logging
    this.shaka_.addEventListener("adaptation", function () {
      const containerTag = me.getContainerTag();
      console.log(`${containerTag} [Shaka] adaptation event fired`);
      me.logVariants(me.shaka_, '(after adaptation)');
    });

    this.shaka_.getNetworkingEngine().registerResponseFilter((type, response) => {
      if (type === shaka.net.NetworkingEngine.RequestType.MANIFEST) {
          const decoder = new TextDecoder('utf-8');
          let manifestText = decoder.decode(response.data);

          manifestText = manifestText.replace(/GROUP-ID="audio-aacl-\d+"/g, 'GROUP-ID="audio-main"');
          manifestText = manifestText.replace(/AUDIO="audio-aacl-\d+"/g, 'AUDIO="audio-main"');
  
          const encoder = new TextEncoder();
          response.data = encoder.encode(manifestText);
      }
  });

    this.shaka_
      .load(src)
      .then(function () {
        me.initShakaMenus();
        
        // 3.3 Log variants at startup
        me.logVariants(me.shaka_, '(at startup)');
        
        // 3.4 Start stats logging (first 30 seconds)
        me.startStatsLogging(me.shaka_);
        
        // Expose instances after source is loaded
        if (me.containerId_) {
          me.exposeShakaInstances();
        }
      })
      .catch(me.retriggerError.bind(this));
  }

  dispose() {
    // Clear stats logging timer
    if (this._statsTimer) {
      clearInterval(this._statsTimer);
      this._statsTimer = null;
    }

    if (this.shaka_) {
      this.shaka_.unload();
      this.shaka_.destroy();
    }
    super.dispose();
  }

  initShakaMenus() {
    // Setup quality tracks, text tracks, audio tracks
    console.log('Setting up Shaka menus...');
    
    try {
      // Setup quality tracks (renditions)
      setupQualityTracks(this, this.shaka_);
      console.log('Quality tracks setup completed');
      
      // Setup text tracks (subtitles)
      setupTextTracks(this, this.shaka_);
      console.log('Text tracks setup completed');
      
      // Setup audio tracks
      setupAudioTracks(this, this.shaka_);
      console.log('Audio tracks setup completed');
    } catch (error) {
      console.error('Error setting up Shaka:', error);
    }
  }

  retriggerError(event) {
    let code;

    // map the shaka player error to the appropriate video.js error
    if (
      event.message &&
      (event.message.indexOf("UNSUPPORTED") > -1 ||
        event.message.indexOf("NOT_SUPPORTED") > -1)
    ) {
      code = 4;
    } else {
      switch (event.category) {
        case 1:
          code = 2;
          break;
        case 2:
        case 3:
        case 4:
          code = 3;
          break;
        case 5:
          code = 1;
          break;
        case 6:
          code = 5;
          break;
        case 7:
        case 8:
        case 9:
          code = 0;
          break;
      }
    }

    this.vjsPlayer.error({
      code,
      message: `${event.code} - ${event.message}`,
    });

    // only reset the shaka player in 10ms async, so that the rest of the
    // calling function finishes
    setTimeout(() => {
      this.dispose();
    }, 10);
  }

  getShaka_() {
    return this.shaka_;
  }

  // 3.3 Log available variants + currently active
  logVariants(player, tag = '') {
    if (!player) return;
    
    try {
      const variants = player.getVariantTracks();
      const rows = variants
        .map(t => ({
          id: t.id,
          bw: t.bandwidth,
          w: t.width,
          h: t.height,
          codecs: t.codecs,
          active: t.active,
          language: t.language,
        }))
        .sort((a, b) => a.bw - b.bw);

      const containerTag = this.getContainerTag();
      console.log(`${containerTag} [Shaka] Variants ${tag}`, rows);
    } catch (error) {
      const containerTag = this.getContainerTag();
      console.warn(`${containerTag} [Shaka] Error logging variants:`, error);
    }
  }

  // 3.4 Log ABR stats periodically during playback (initial 30 seconds)
  startStatsLogging(player) {
    if (!player) return;
    
    // Clear any existing timer
    if (this._statsTimer) {
      clearInterval(this._statsTimer);
      this._statsTimer = null;
    }

    const containerTag = this.getContainerTag();
    let statsCount = 0;
    const maxStatsLogs = 15; // 30 seconds / 2 seconds = 15 logs

    this._statsTimer = setInterval(() => {
      try {
        const s = player.getStats();
        console.log(`${containerTag} [Shaka] Stats`, {
          estBw: s.estimatedBandwidth,
          playTime: s.playTime,
          bufferingTime: s.bufferingTime,
          variantBw: s.streamBandwidth,
          droppedFrames: s.droppedFrames,
        });

        statsCount++;
        if (statsCount >= maxStatsLogs) {
          clearInterval(this._statsTimer);
          this._statsTimer = null;
        }
      } catch (error) {
        const containerTag = this.getContainerTag();
        console.warn(`${containerTag} [Shaka] Error getting stats:`, error);
        clearInterval(this._statsTimer);
        this._statsTimer = null;
      }
    }, 2000);
  }

  // Set container ID for exposure
  setContainerId(containerId) {
    this.containerId_ = containerId;
    // If Shaka is already initialized, expose immediately
    if (this.shaka_) {
      this.exposeShakaInstances();
    }
  }

  // Expose Shaka instances to window with container-id as key
  exposeShakaInstances() {
    if (typeof window === "undefined" || !this.containerId_ || !this.shaka_) {
      return;
    }

    const containerId = this.containerId_;

    // Initialize window.__shaka_players if not exists
    if (!window.__shaka_players) {
      window.__shaka_players = {};
    }
    if (!window.__shaka_videos) {
      window.__shaka_videos = {};
    }
    if (!window.__shaka_debug) {
      window.__shaka_debug = {};
    }

    // Expose instances with container-id as key
    window.__shaka_players[containerId] = this.shaka_;
    window.__shaka_videos[containerId] = this.el_;

    // Expose debug helpers
    window.__shaka_debug[containerId] = {
      player: this.shaka_,
      video: this.el_,
      tech: this,
      vjsPlayer: this.vjsPlayer,
      getManifest: () => this.shaka_.getManifest(),
      getConfiguration: () => this.shaka_.getConfiguration(),
      getStats: () => this.shaka_.getStats(),
      getBufferedInfo: () => this.shaka_.getBufferedInfo(),
      getPlaybackRate: () => this.shaka_.getPlaybackRate(),
      getVolume: () => this.shaka_.getVolume(),
      isLive: () => this.shaka_.isLive(),
      seekRange: () => this.shaka_.seekRange(),
    };

    // Log exposure
    console.log(`[Shaka] Exposed instances for container "${containerId}":`, {
      __shaka: !!window.__shaka,
      __shaka_players: !!window.__shaka_players[containerId],
      __shaka_videos: !!window.__shaka_videos[containerId],
      __shaka_debug: !!window.__shaka_debug[containerId],
    });
  }
}

export default Shaka;
